﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using Thomas_Erdoesi.Game_Analyzer;
using Thomas_Erdoesi.Game_Objects;
using Thomas_Erdoesi.Properties;

namespace Thomas_Erdoesi.Game_Player
{
    public class Player
    {

        public static List<GameObject> Log = new List<GameObject>();

        public void PlayGame(GameStatus Status)
        {
            // Spielen, sofern Raumschiff sichtbar ist
            if (Status.Spaceship != null)
            {
                // Positionen für (erwarteten) nächsten Frame vorhersagen
                if (Status.Spaceship != null)
                    Status.Spaceship.PredictLocation(Settings.Default.EstimatedInputLatency);

                if (Status.Saucer != null)
                    Status.Saucer.PredictLocation(Settings.Default.EstimatedInputLatency);

                foreach (Asteroid CurrentAsteroid in Status.Asteroids)
                    CurrentAsteroid.PredictLocation(Settings.Default.EstimatedInputLatency);

                // Spielstrategie

                // Mit höchster Priorität Kollisionen verhindern
                if (AvoidCollision(Status)) return;

                // Aktuelles Ziel ermitteln
                GameObject Target = IdentifyPrimaryTarget(Status);

                // Reaktion festlegen
                if (Target != null)
                {
                    EngagePrimaryTarget(Status, Target);
                    ShootAtTarget(Status);
                }
                else
                {
                    MoveToCenter(Status);
                }
            }
        }

        /// <summary>
        /// Primäres Ziel identifizieren
        /// </summary>
        /// <param name="Status"></param>
        /// <returns></returns>
        private GameObject IdentifyPrimaryTarget(GameStatus Status)
        {
            GameObject Target = null;

            double TargetScore = double.MinValue;

            // UFO hat absolute Zielpriorität, sofern es sich in Schussweite befindet
            if (Status.Saucer != null)
            {
                GameObjectDistance Distance = Status.Spaceship.GetPredictedDistance(Status.Saucer);

                if (Distance.ObjectDistance < Settings.Default.EstimatedShotRange)
                    return Status.Saucer;
                else
                {
                    TargetScore = CalculateTargetScore(Status, Status.Saucer);
                    Target = Status.Saucer;
                }
            }

            // Asteroiden bewerten
            foreach (Asteroid CurrentAsteroid in Status.Asteroids)
            {
                if (CurrentAsteroid.IncomingShots < CurrentAsteroid.Size / 10 + Settings.Default.ShotsPerTarget)
                {
                    double Score = CalculateTargetScore(Status, CurrentAsteroid);

                    if (Score > TargetScore)
                    {
                        TargetScore = Score;
                        Target = CurrentAsteroid;
                    }
                }
            }

            if (Target == null)
            {
                foreach (Asteroid CurrentAsteroid in Status.Asteroids)
                {
                    if (CurrentAsteroid.IncomingShots >= CurrentAsteroid.Size / 10 + Settings.Default.ShotsPerTarget)
                    {
                        double Score = CalculateTargetScore(Status, CurrentAsteroid);

                        if (Score > TargetScore)
                        {
                            TargetScore = Score;
                            Target = CurrentAsteroid;
                        }
                    }
                }
            }

            return Target;
        }

        /// <summary>
        /// Wertung für ein mögliches Ziel berechnen
        /// </summary>
        /// <param name="Status"></param>
        /// <param name="Target"></param>
        /// <returns></returns>
        private double CalculateTargetScore(GameStatus Status, GameObject Target)
        {
            double Result = 0;

            // Vorhersage für frühest möglichen Schusszeitpunkt berechnen
            Status.Spaceship.PredictLocation(2);
            Target.PredictLocation(5);

            // Abstand berechnen
            GameObjectDistance CurrDistance = Status.Spaceship.GetDistance(Target);
            GameObjectDistance PredDistance = Status.Spaceship.GetPredictedDistance(Target);

            // Notwendige Drehung (zum Zeitpunkt des übernächsten Frames) berechnen
            double PredDiffAngle;

            if (Target.Speed.SpeedSqared > 0)
            {
                GameObjectLocation InitialShotLocation = GameMath.CalculateInitialShotLocation(Status.Spaceship.PredictedLocation, Status.Spaceship.PredictedAngle);

                GameObjectSpeed TargetInterceptionVector = GameMath.CalculateInterceptionSpeed(InitialShotLocation,
                                                                                               Status.Spaceship.Speed,
                                                                                               Target.PredictedLocation,
                                                                                               Target.Speed,
                                                                                               Settings.Default.EstimatedShotSpeed);

                PredDiffAngle = GameMath.AngleDiff(Math.Atan2(TargetInterceptionVector.sx, TargetInterceptionVector.sy), Status.Spaceship.PredictedAngle);
            }
            else
                PredDiffAngle = Math.PI;

            // Basiswertung aufgrund der Entfernung berechnen
            Result -= Settings.Default.TargetDistanceBonusFactor * PredDistance.ObjectDistance;
               
            // Bonus für große Objekte
            Result += Settings.Default.TargetSizeBonusFactor * Target.Size;

            // Bonus für Objekte die keine große Kurskorrektur erfordern
            Result -= Settings.Default.TargetAtCurrentAngleBonusFactor * Math.Abs(PredDiffAngle);

            // Bonus für sich nähernde Objekte berechnen
            Result += Settings.Default.TargetGettingCloserBonusFactor * (CurrDistance.ObjectDistance - PredDistance.ObjectDistance);

            // Bonus für erreichbare Objekte berücksichtigen
            if (PredDistance.ObjectDistance < Settings.Default.EstimatedShotRange)
            {
                Result += Settings.Default.TargetWithinShotRangeBonus;
            }

            // Bonus für gefährliche Objekte
            if (Target.TimeToCollision < Settings.Default.TimeToCollisionBonusThreshold)
            {
                Result += Settings.Default.TimeToCollisionBonusFactor * (Settings.Default.TimeToCollisionBonusThreshold - Target.TimeToCollision);
            }

            return Result;
        }

        /// <summary>
        /// Schießen, sofern ein Ziel getroffen werden kann
        /// </summary>
        /// <param name="Status"></param>
        private void ShootAtTarget(GameStatus Status)
        {
            // Dauerfeuer auf letzten Asteroiden oder UFO 
            if (Status.Asteroids.Count == 1 || Status.Saucer != null)
            {
                Status.Keys.Fire = true;
                return;
            }

            // Schießen, wann immer etwas getroffen werden kann
            double MinTTC = double.MaxValue;

            Status.Spaceship.PredictLocation(Settings.Default.SpaceshipPredictionPeriod);

            Shot TestShot = new Shot(Status, Status.Spaceship.Location.x, Status.Spaceship.Location.y);

            TestShot.Speed = GameMath.CalculateShotSpeed(Status.Spaceship.Speed, Status.Spaceship.Angle);

            foreach (Asteroid TargetObject in Status.Asteroids)
            {
                TargetObject.PredictLocation(Settings.Default.TargetPredictionPeriod);

                double TTC = GameMath.CalculateTimeToCollision(TestShot.Location, TestShot.Speed, TestShot.Size,
                                                               TargetObject.Location, TargetObject.Speed, TargetObject.Size);
                if (TTC < MinTTC) MinTTC = TTC;
            }

            if (Status.Saucer != null)
            {
                Status.Saucer.PredictLocation(Settings.Default.TargetPredictionPeriod);

                double TTC = GameMath.CalculateTimeToCollision(TestShot.Location, TestShot.Speed, TestShot.Size,
                                                               Status.Saucer.Location, Status.Saucer.Speed, Status.Saucer.Size);
                if (TTC < MinTTC) MinTTC = TTC;
            }

            if (MinTTC <= Settings.Default.EstimatedShotRange / Settings.Default.EstimatedShotSpeed)
                Status.Keys.Fire = true;
        }

        /// <summary>
        /// Raumschiff zum primären Ziel drehen
        /// </summary>
        /// <param name="Status"></param>
        /// <param name="Target"></param>
        private void EngagePrimaryTarget(GameStatus Status, GameObject Target)
        {
            GameObjectDistance CurrDistance = Status.Spaceship.GetDistance(Target);
            GameObjectDistance PredDistance = Status.Spaceship.GetPredictedDistance(Target);

            GameObjectSpeed TargetInterceptionVektor = null;

            // Da eine Reaktion frühestens in übernächsten Frame erfolgen kann, muss
            // zunächst die dann vorlegende Basissituation vorhergesagt werden
            // Die Vorhersage wird nur für das Raumschiff und das Zielobjekt ermittelt
            Status.Spaceship.PredictLocation(Settings.Default.SpaceshipPredictionPeriod);
            Target.PredictLocation(Settings.Default.TargetPredictionPeriod);

            // Nun wird der Geschwindigkeitsvektor (ist auch der Richtungsvektor) für
            // einen Schuss berechnet, der genau mit dem Objekt kollidiert, sofern es
            // seine ermittelte Bewegungsrichtung fortsetzt
            TargetInterceptionVektor = GameMath.CalculateInterceptionSpeed(Status.Spaceship.Location,
                                                                           Status.Spaceship.Speed,
                                                                           Target.PredictedLocation,
                                                                           Target.Speed,
                                                                           Settings.Default.EstimatedShotSpeed);

            // Schiff in Abfangrichtung auf das nächstgelegene Objekt drehen
            double DiffAngle = 0;

            if (TargetInterceptionVektor.sx != 0 || TargetInterceptionVektor.sy != 0)
                DiffAngle = GameMath.AngleDiff(Status.Spaceship.PredictedAngle, Math.Atan2(TargetInterceptionVektor.sy, TargetInterceptionVektor.sx));
            else
                DiffAngle = GameMath.AngleDiff(Status.Spaceship.PredictedAngle, Math.Atan2(PredDistance.dy, PredDistance.dx));

            if (DiffAngle > 0) Status.Keys.Left = true;
            else if (DiffAngle < 0) Status.Keys.Right = true;

            // Beschleunigen, wenn Ziel zu weit entfernt und kein UFO
            if (Target != Status.Saucer &&
                Status.Spaceship.Speed.Speed < Settings.Default.SpaceshipSpeedLimit &&
                (PredDistance.ObjectDistance > Settings.Default.EstimatedShotRange ||
                 PredDistance.ObjectDistance > Settings.Default.FollowTargetThreshold &&
                 PredDistance.ObjectDistance > CurrDistance.ObjectDistance))
                Status.Keys.Thrust = true;
        }

        /// <summary>
        /// Zum Zentrum bewegen
        /// </summary>
        /// <param name="Status"></param>
        private void MoveToCenter(GameStatus Status)
        {
            GameObjectLocation Center = new GameObjectLocation(GameConstants.CenterX, GameConstants.CenterY);
            GameObjectDistance ToCenter = new GameObjectDistance(Status.Spaceship.Location, Center);

            if (ToCenter.ObjectDistance > 100)
            {
                double DiffAngle = GameMath.AngleDiff(Status.Spaceship.Angle, Math.Atan2(ToCenter.dy, ToCenter.dx));

                if (DiffAngle > 0) Status.Keys.Left = true;
                else if (DiffAngle < 0) Status.Keys.Right = true;

                if (Status.Spaceship.Speed.Speed < ToCenter.ObjectDistance / 200) Status.Keys.Thrust = true;
            }
        }

        /// <summary>
        /// Kollisionen verhindern
        /// </summary>
        private bool AvoidCollision(GameStatus Status)
        {
            double MinTtc = double.MaxValue;
            GameObject MinTtcObj = null;

            Log.Add(Status.Spaceship);

            foreach (Asteroid CurrentAsteroid in Status.Asteroids)
            {
                Log.Add(CurrentAsteroid);

                CurrentAsteroid.CalculateTimeToCollision(Status.Spaceship);

                if (CurrentAsteroid.TimeToCollision < MinTtc)
                {
                    MinTtc = CurrentAsteroid.TimeToCollision;
                    MinTtcObj = CurrentAsteroid;
                }
            }

            foreach (Shot CurrentShot in Status.Shots)
            {
                CurrentShot.CalculateTimeToCollision(Status.Spaceship);

                if (CurrentShot.TimeToCollision < MinTtc)
                {
                    MinTtc = CurrentShot.TimeToCollision;
                    MinTtcObj = CurrentShot;
                }
            }

            if (Status.Saucer != null)
            {
                Status.Saucer.CalculateTimeToCollision(Status.Spaceship);

                if (Status.Saucer.TimeToCollision < MinTtc)
                {
                    MinTtc = Status.Saucer.TimeToCollision;
                    MinTtcObj = Status.Saucer;
                }
            }

            if (MinTtc < Settings.Default.HyperspaceTimeToCollisionThreshold)
            {
                Status.Keys.Hyperspace = true;
                Status.Keys.Fire = true;
                return true;
            }

            // TODO: Evtl. entfernen von gefährlichen Objekten

            return false;
        }
    }
}